//
//  UKPluginManager.m
//  PluginApp
//
//  Created by Uli Kusterer on 28.06.09.
//  Copyright 2009 The Void Software. All rights reserved.
//

// -----------------------------------------------------------------------------
//	Headers:
// -----------------------------------------------------------------------------

#import "UKPluginManager.h"
#import <CoreServices/CoreServices.h>


// -----------------------------------------------------------------------------
//	Constants:
// -----------------------------------------------------------------------------

NSString*	UKPluginManagerPluginsFolderName = @"PlugIns";
NSString*	UKPluginManagerExportFileExtensionKey = @"UKExportFileExtension";


// -----------------------------------------------------------------------------
//	Globals:
// -----------------------------------------------------------------------------

static UKPluginManager*	sSharedPluginManager = nil;


@implementation UKPluginManager

// -----------------------------------------------------------------------------
//	Singleton accessor:
// -----------------------------------------------------------------------------

+(UKPluginManager*)	sharedPluginManager
{
	if( !sSharedPluginManager )
	{
		[[UKPluginManager alloc] init];	// Sets sSharedPluginManager internally.
		[sSharedPluginManager loadPlugins];
	}
	
	return sSharedPluginManager;
}


// -----------------------------------------------------------------------------
//	* CONSTRUCTOR
// -----------------------------------------------------------------------------

-(id)	init
{
	if(( self = [super init] ))
	{
		if( sSharedPluginManager )	// We only allow one instance.
		{
			[self autorelease];
			return nil;
		}
		
		sSharedPluginManager = self;	// Do this first, so we can do other stuff and no calls we trigger accidentally create a new instance.
		
		plugins = [[NSMutableDictionary alloc] init];
	}
	
	return self;
}


// -----------------------------------------------------------------------------
//	* DESTRUCTOR
// -----------------------------------------------------------------------------

-(void)	dealloc
{
	[plugins release];
	plugins = nil;

	if( sSharedPluginManager && sSharedPluginManager == self )
		sSharedPluginManager = nil;
	
	[super dealloc];
}


// -----------------------------------------------------------------------------
//	loadPlugins
//		Called by +sharedPluginManager to lazily load our plugins.
// -----------------------------------------------------------------------------

-(void)	loadPlugins
{
	NSString*		appName = [[[NSBundle mainBundle] infoDictionary] objectForKey: @"CFBundleExecutable"];
	NSString*		pluginsSubpath = [appName stringByAppendingPathComponent: UKPluginManagerPluginsFolderName];
	
	// Get all App support folders, in ~/Library/, /Library/, /System/Library/ etc.:
	NSArray*		appSupPaths = NSSearchPathForDirectoriesInDomains( NSApplicationSupportDirectory, NSAllDomainsMask, YES );
	
	// Now build paths for our "PluginApp/Plug-ins/" folders in those:
	NSMutableArray*	paths = [NSMutableArray array];
	for( NSString* currAppSupPath in appSupPaths )
		[paths addObject: [currAppSupPath stringByAppendingPathComponent: pluginsSubpath]];
	
	// Add an additional entry for PluginApp.app/Contents/Plug-ins/:
	//	That way we can provide standard plugins.
	NSString*		appBundlePluginsFolder = [[NSBundle mainBundle] builtInPlugInsPath];
	[paths addObject: appBundlePluginsFolder];
	
	// Now loop over all those paths and load plugins from there, if they exist:
	for( NSString* currFolder in paths )
	{
		BOOL	isDirectory = NO;
		if( [[NSFileManager defaultManager] fileExistsAtPath: currFolder isDirectory: &isDirectory] && isDirectory )
			[self loadPluginsFromFolder: currFolder];
	}
}


// -----------------------------------------------------------------------------
//	loadPluginsFromFolder:
//		Called by -loadPlugins to actually load all the plugins in one folder.
// -----------------------------------------------------------------------------

-(void)	loadPluginsFromFolder: (NSString*)folderPath
{
	NSDirectoryEnumerator*	dirEnny = [[NSFileManager defaultManager] enumeratorAtPath: folderPath];
	NSString*				currPluginName = nil;
	while(( currPluginName = [dirEnny nextObject] ))
	{
		[dirEnny skipDescendents];	// Don't have dirEnny scan subfolders and package contents.
		
		// Ignore Unix-style invisible files:
		if( [currPluginName characterAtIndex: 0] == '.' )
			continue;
		
		// Only accept files with currect suffix:
		if( ![[currPluginName pathExtension] isEqualToString: @"paPlugin"] )
			continue;
		
		// Also, ignore files with HFS 'invisible' bit on:
		NSString*			currPluginPath = [folderPath stringByAppendingPathComponent: currPluginName];
		LSItemInfoRecord	infoRec = { 0 };
		OSStatus err = LSCopyItemInfoForURL( (CFURLRef) [NSURL fileURLWithPath: currPluginPath], kLSRequestBasicFlagsOnly, &infoRec );
		if( err != noErr || (infoRec.flags & kLSItemInfoIsInvisible) )
			continue;
		
		// Now all conditions have been checked, load it:
		NSBundle*	pluginBundle = [NSBundle bundleWithPath: currPluginPath];
		NSString*	fileExtension = [[pluginBundle infoDictionary] objectForKey: UKPluginManagerExportFileExtensionKey];
		
		if( fileExtension && [fileExtension isKindOfClass: [NSString class]]
			&& ![fileExtension isEqualToString: @""] )
		{
			[plugins setObject: pluginBundle forKey: [fileExtension lowercaseString]];
		}
	}
}


// -----------------------------------------------------------------------------
//	availablePluginExtensions:
//		Return an array of filename suffixes representing the types of files
//		plugins can generate.
// -----------------------------------------------------------------------------

-(NSArray*)	availablePluginExtensions
{
	return [plugins allKeys];
}


// -----------------------------------------------------------------------------
//	pluginClassForExtension:
//		Takes a filename suffixes and returns the Class for the plugin that
//		exports to files with that extension. Returns Nil if there is no
//		matching plugin.
// -----------------------------------------------------------------------------

-(Class)	pluginClassForExtension: (NSString*)fileExtension
{
	NSBundle*	pluginBundle = [plugins objectForKey: [fileExtension lowercaseString]];
	if( !pluginBundle )
		return Nil;
	else
		return [pluginBundle principalClass];
}

@end
