/* =============================================================================
    PROJECT:    Filie
    FILE:       UKSidebarDataSource.m
    
    COPYRIGHT:  (c) 2004 by M. Uli Kusterer, all rights reserved.
    
    AUTHORS:    M. Uli Kusterer - UK
    
    LICENSES:   GNU GPL
    
    REVISIONS:
        2004-12-10  UK  Created.
   ========================================================================== */

// -----------------------------------------------------------------------------
//  Headers:
// -----------------------------------------------------------------------------

#import "UKSidebarDataSource.h"
#import "UKDirectoryEnumerator.h"
#import "IconKit.h"
#import "NDAlias.h"
#import "NSWorkspace+PreviewFile.h"
#import "NSImage+NiceScaling.h"


@implementation UKSidebarDataSource

// -----------------------------------------------------------------------------
//  reload:
//      Cause the data source to tell the delegate about the currently
//      available files. This is the main method you use to kick of processing.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void) reload: (id)sender
{
    [delegate dataSourceWillReload: self];
        
    NSString*               filePath = nil;
	NSMutableDictionary*    attrs = nil;
    	
    // Add "Network" icon.
    attrs = [[[[NSFileManager defaultManager] fileAttributesAtPath: @"/Network" traverseLink: NO] mutableCopy] autorelease];
    [delegate listItem: [NSURL fileURLWithPath: @"/Network"] withAttributes: attrs source: self];

	// Now list all mounted volumes:
	NSEnumerator*   di = [[[NSWorkspace sharedWorkspace] mountedLocalVolumePaths] objectEnumerator];
	
	while( (filePath = [di nextObject]) )
	{
        if( [filePath hasPrefix: @"/Users/"] )          // Skip mounted FileVault user directories.
            continue; 
        if( [filePath isEqualToString: @"/Network"] )   // We add network manually, coz otherwise it doesn't always show up.
            continue; 
        
        NSMutableDictionary*   attrs = [[[[NSFileManager defaultManager] fileAttributesAtPath: filePath traverseLink: NO] mutableCopy] autorelease];
        [delegate listItem: [NSURL fileURLWithPath: filePath] withAttributes: attrs source: self];
	}
    
    [delegate dataSourceWillRecache: self];
    
	// Now list user-defined items:
	NSDictionary*   sidebarPlist = [[NSUserDefaults standardUserDefaults] persistentDomainForName: @"com.apple.sidebarlists"];
	NSDictionary*   userItems = [sidebarPlist objectForKey: @"useritems"];
	NSArray*		customListItems = [userItems objectForKey: @"CustomListItems"];
	NSDictionary*   currItem = nil;
	NSEnumerator*   enny = [customListItems objectEnumerator];
	
	while( (currItem = [enny nextObject]) )
	{
        NS_DURING
            NSData*     aliasHandleData = [currItem objectForKey: @"Alias"];
            filePath = [[NDAlias aliasWithData: aliasHandleData] path];
            NSMutableDictionary*   attrs = [[[[NSFileManager defaultManager] fileAttributesAtPath: filePath traverseLink: NO] mutableCopy] autorelease];
            [delegate listItem: [NSURL fileURLWithPath: filePath] withAttributes: attrs source: self];
        NS_HANDLER
            NSLog( @"Couldn't load sidebar item: %@", localException );
        NS_ENDHANDLER
	}
    
    [delegate dataSourceDidReload: self];
}


// -----------------------------------------------------------------------------
//  delegate:
//      Who's your daddy?
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(id)               delegate
{
    return delegate;
}


// -----------------------------------------------------------------------------
//  setDelegate:
//      Specify the object to send our delegate messages to.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void)             setDelegate: (id)del
{
    delegate = del;
}


// -----------------------------------------------------------------------------
//  placeholderIconForItem:attributes:
//      Return an image to display as the icon for the item at the specified
//      URL with the specified attributes.
//
//      This needn't be an exact icon, however, it must be one that can be
//      obtained *very* fast. For icons that have to be downloaded or generated,
//      return a generic icon here and do the actual work in iconForItem: to get
//      a second, more accurate icon.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(NSImage*)         placeholderIconForItem: (NSURL*)url attributes: (NSDictionary*)attrs
{
    if( [[attrs objectForKey: NSFileType] isEqualToString: NSFileTypeDirectory] )
        return [[IKIcon iconWithIdentifier: IKIconGenericFolder] image];
    else
        return [[IKIcon iconWithIdentifier: IKIconGenericDocument] image];
}


// -----------------------------------------------------------------------------
//  iconForItem:attributes:
//      Return the icon for the item at the specified URL with the specified
//      attributes.
//
//      This is the icon the file will have once everything's been loaded. This
//      method may be a tad slower, as it's called from a second thread while
//      the user can still work on.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(NSImage*)         iconForItem: (NSURL*)url attributes: (NSDictionary*)attributes
{
    NSString*       path = [url path];
	NSImage*        fileIcon = nil;
    NSString*       suf = [[path pathExtension] lowercaseString];
    NSArray*        types = nil;
    NSSize          iconSize = NSMakeSize(128,128);
    BOOL            hasCustomIcon = !attributes || [[attributes objectForKey: UKItemHasCustomIcon] boolValue];
    
    if( [delegate showIconPreview] )
        types = [NSImage imageFileTypes];
    
    if( types )
        fileIcon = [[NSWorkspace sharedWorkspace] previewImageForFile: path size: iconSize];
    
    // Couldn't generate a preview, although we want icon previews, and no custom icon?
    if( !fileIcon && types && [types containsObject: suf] && (!hasCustomIcon || !attributes) )
    {
        // Load the full file and manually cobble together a preview:
        fileIcon = [[[NSImage alloc] initWithContentsOfFile: path] autorelease];
        //fileIcon = [fileIcon scaledImageToFitSize: iconSize];
    }
    
    if( !fileIcon )    // Still no icon?
        fileIcon = [[IKIcon iconForFile: path] image];  // Ask Icon Services for an icon.
    
    return fileIcon;
}


@end
