/*
 *  IKIcon.m
 *  
 *
 *  Created by Uli Kusterer on 31.12.04.
 *  Copyright 2004 M. Uli Kusterer. All rights reserved.
 *
 */

// -----------------------------------------------------------------------------
//  Headers:
// -----------------------------------------------------------------------------

#include <Carbon/Carbon.h>
#import "IKIcon.h"
#import "IKIconPrivate.h"
#import "NSString+CarbonUtilities.h"
#import "NSImage+IconRef.h"
#import "IKIconRefImageRep.h"
#import "UKDebugNames.h"


// -----------------------------------------------------------------------------
//  Globals:
// -----------------------------------------------------------------------------

static NSMutableDictionary*     sIKIconObjectCache = nil;


// -----------------------------------------------------------------------------
//  Constants:
// -----------------------------------------------------------------------------

IKIconIdentifier    IKIconGenericDocument = { kSystemIconsCreator, kGenericDocumentIcon };
IKIconIdentifier    IKIconGenericApplication = { kSystemIconsCreator, kGenericApplicationIcon };
IKIconIdentifier    IKIconGenericPlugIn = { kSystemIconsCreator, kGenericExtensionIcon };    // FIX ME! Make a plugin icon!
IKIconIdentifier    IKIconGenericFolder = { kSystemIconsCreator, kGenericFolderIcon };
IKIconIdentifier    IKIconPrivateFolder = { kSystemIconsCreator, kPrivateFolderIcon };
IKIconIdentifier    IKIconWriteOnlyFolder = { kSystemIconsCreator, kDropFolderIcon };
IKIconIdentifier    IKIconRecyclerFolder = { kSystemIconsCreator, kTrashIcon };
IKIconIdentifier    IKIconRecyclerFolderFull = { kSystemIconsCreator, kFullTrashIcon };
// ...
IKIconIdentifier    IKIconLinkBadge = { kSystemIconsCreator, kAliasBadgeIcon };
IKIconIdentifier    IKIconLockedBadge = { kSystemIconsCreator, kLockedBadgeIcon };
IKIconIdentifier    IKIconScriptBadge = { kSystemIconsCreator, kAppleScriptBadgeIcon };
IKIconIdentifier    IKIconReadOnlyBadge = { kSystemIconsCreator, kSharingPrivsReadOnlyIcon };
IKIconIdentifier    IKIconWriteOnlyBadge = { kSystemIconsCreator, kSharingPrivsWritableIcon };

// System icons (not for files):
IKIconIdentifier    IKIconAlertNote = { kSystemIconsCreator, kAlertNoteIcon };
IKIconIdentifier    IKIconAlertWarning = { kSystemIconsCreator, kAlertCautionIcon };
IKIconIdentifier    IKIconAlertFailure = { kSystemIconsCreator, kAlertStopIcon };

// Notifications:
NSString*           IKIconChangedNotification = @"IKIconChangedNotification";  // Sent with the IKIcon as the object whenever update is called.


// -----------------------------------------------------------------------------
//  Implementation:
// -----------------------------------------------------------------------------

@implementation IKIcon (IKIconMacPrivate)

// -----------------------------------------------------------------------------
//  initialize:
//      Create our cache when this class is loaded.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

+(void) initialize
{
    [super initialize];
    
    if( !sIKIconObjectCache )
        sIKIconObjectCache = [[NSMutableDictionary alloc] init];
}


// -----------------------------------------------------------------------------
//  fileIconFromCache:
//      Find an object for the specified IconRef. Returns NIL if there is no
//      object for that icon ref yet. That way we don't have dozens of objects
//      wrapping around the same IconRef.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

+(IKIcon*)  fileIconFromCache: (IconRef)theIcon
{
    IKIcon*     outIconObj = nil;
    
    @synchronized( sIKIconObjectCache )
    {
        NSValue*    ref = [NSValue valueWithPointer: (void*) theIcon];
        outIconObj = [sIKIconObjectCache objectForKey: ref];
    }
    return outIconObj;
}


// -----------------------------------------------------------------------------
//  addIconObjectToCache:
//      Add a new object to our cache so fileIconFromCache: knows about it.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

+(void)         addIconObjectToCache: (IKIcon*)iconObj
{
    @synchronized( sIKIconObjectCache )
    {
        NSValue*    ref = [NSValue valueWithPointer: (void*) [iconObj iconRef]];
        [sIKIconObjectCache setObject: iconObj forKey: ref];
    }
}


// -----------------------------------------------------------------------------
//  removeIconFromCache:
//      Remove an object from our cache so fileIconFromCache: no longer returns
//      it.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

+(void)         removeIconFromCache: (IconRef)theIcon
{
    @synchronized( sIKIconObjectCache )
    {
        NSValue*    ref = [NSValue valueWithPointer: (void*) theIcon];
        [sIKIconObjectCache removeObjectForKey: ref];
    }
}

// -----------------------------------------------------------------------------
//  initWithType:creator:extension:mimeType:
//      Return an icon for a particular kind of file. This may give you a cached
//      object instead of the one you alloced originally.
//
//      * MAC-SPECIFIC *
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(id)   initWithType: (OSType)typ creator: (OSType)creator
                extension: (NSString*)ext mimeType: (NSString*)mime
{
    self = [super init];
    if( !self )
        return nil;
    
    @synchronized( sIKIconObjectCache ) // We don't access the cache directly, but this makes sure we don't call into Icon Services from several threads at once.
    {
        OSErr err = noErr;
        err = GetIconRefFromTypeInfo( creator, typ, (CFStringRef) ext,
                                        (CFStringRef) mime,
                                        kIconServicesNormalUsageFlag,
                                        &iconRef);
        if( err != noErr )
        {
            NSLog(@"IKIcon initWithType: MacOS error code=%d",err);
            [self autorelease];
            return nil;
        }
        
        IKIcon* cachedIcon = [[self class] fileIconFromCache: iconRef];
        if( cachedIcon )
        {
            [self autorelease];
            return [cachedIcon retain];
        }
        else
            [[self class] addIconObjectToCache: self];
    }
    
    return self;
}


// -----------------------------------------------------------------------------
//  initWithIconRef:
//      Return an icon for a particular IconRef. This may give you a cached
//      object instead of the one you alloced originally.
//      
//      Takes over the IconRef passed in.
//
//  REVISIONS:
//      2005-02-09  UK  Created.
// -----------------------------------------------------------------------------

-(id)   initWithIconRef: (IconRef)ref
{
    self = [super init];
    if( !self )
        return nil;
    
    @synchronized( sIKIconObjectCache ) // We don't access the cache directly, but this makes sure we don't call into Icon Services from several threads at once.
    {
        iconRef = ref;  // Makes sure that if a cache happens, this object will release the IconRef again.
        
        IKIcon* cachedIcon = [[self class] fileIconFromCache: iconRef];
        if( cachedIcon )
        {
            [self autorelease];
            return [cachedIcon retain];
        }
        else
            [[self class] addIconObjectToCache: self];
    }
    
    return self;
}


// -----------------------------------------------------------------------------
//  drawInRect:
//      This performs the actual drawing. It doesn't go through the cached
//      NSImage but rather directly draws the icon ref, which scales more
//      beautifully.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void) drawInRect: (NSRect)rect
{
    CGContextRef    con = (CGContextRef) [[NSGraphicsContext currentContext] graphicsPort];
    RGBColor        labelColor = { 0, 0, 0 };
    
    @synchronized( sIKIconObjectCache ) // We don't access the cache directly, but this makes sure we don't call into Icon Services from several threads at once.
    {
        (OSStatus) PlotIconRefInContext( con, (const CGRect*) &rect, kAlignNone, kTransformNone,
                                            &labelColor, kPlotIconRefNormalFlags, iconRef );
    }
}


-(IconRef)  iconRef
{
    return iconRef;
}


@end

@implementation IKIcon

// -----------------------------------------------------------------------------
//  Convenience Factory Methods:
// -----------------------------------------------------------------------------

+(id)       iconForFile: (NSString*)fpath
{
    return [[(IKIcon*)[self alloc] initForFile: fpath] autorelease];
}


+(id)       iconWithIdentifier: (IKIconIdentifier)identifier
{
    return [[(IKIcon*)[self alloc] initWithIdentifier: identifier] autorelease];
}


+(id)       iconWithExtension: (NSString*)suffix mimeType: (NSString*)mime
                attributes: (NSDictionary*)dict
{
    return [[[self alloc] initWithExtension: suffix mimeType: mime attributes: dict] autorelease];
}


+(id)       iconWithSize: (NSSize)size
{
    return [[[self alloc] initWithSize: size] autorelease];
}

+(id)       iconWithImage: (NSImage*)image
{
    return [[[self alloc] initWithImage: image] autorelease];
}


// -----------------------------------------------------------------------------
//  Constructors:
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
//  initForFile:
//      Return an icon for a particular file. This may give you a cached
//      object instead of the one you alloced originally.
//
//      TODO: We could probably write a variant of this that takes a
//      fileAttributes dictionary or FSCatalogInfo struct. That would be faster
//      in cases where we're querying the attributes anyway.
//
//  REVISIONS:
//      2005-02-09  UK  Renamed to initForFile: to make clear it gets the icon
//                      *for* the file, not *from* it.
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(id)   initForFile: (NSString*)fpath
{
    self = [super init];
    if( !self )
        return nil;
    
    @synchronized( sIKIconObjectCache ) // We don't access the cache directly, but this makes sure we don't call into Icon Services from several threads at once.
    {
        FSRef       fileRef;
        SInt16      iconLabel = 0;
        OSStatus    err = noErr;
        
        if( ![fpath getFSRef: &fileRef] )
        {
            NSLog(@"IKIcon initForFile: File %@ doesn't exist.", fpath);
            [self autorelease];
            return nil;
        }
        
        err = GetIconRefFromFileInfo( &fileRef, 0, NULL,
                                        kFSCatInfoNone, NULL,
                                        kIconServicesNormalUsageFlag,
                                        &iconRef,
                                        &iconLabel );
        if( err != noErr )
        {
            NSLog(@"IKIcon initForFile: MacOS error code=%ld",err);
            [self autorelease];
            return nil;
        }
        
        IKIcon* cachedIcon = [[self class] fileIconFromCache: iconRef];
        if( cachedIcon )
        {
            dummy = YES;
            [self autorelease];
            return [cachedIcon retain];
        }
        else
            [[self class] addIconObjectToCache: self];
    }
    
    return self;
}


-(id)       initWithIdentifier: (IKIconIdentifier)identifier
{
    return [self initWithType: identifier.type creator: identifier.creator
                extension: nil mimeType: nil];
}


-(id)       initWithExtension: (NSString*)suffix mimeType: (NSString*)mime
                attributes: (NSDictionary*)dict
{
    NSNumber*   typeNum = [dict objectForKey: NSFileHFSTypeCode];
    NSNumber*   creaNum = [dict objectForKey: NSFileHFSCreatorCode];
    OSType      type = typeNum ? [typeNum unsignedIntValue] : 0L;
    OSType      crea = creaNum ? [creaNum unsignedIntValue] : 0L;

    return [self initWithType: type creator: crea
                extension: suffix mimeType: mime];
}

-(id)       initWithSize: (NSSize)size
{
    return [self initWithImage: [[[NSImage alloc] initWithSize: size] autorelease]];
}

-(id)       initWithImage: (NSImage*)image
{
    self = [super init];
    if( !self )
        return nil;
    
    iconRef = [image iconRefRepresentation];
    cachedImage = nil;
    
    return self;
}

-(id)       initWithDictionary: (NSDictionary*)plist
{
    return nil;
}


// -----------------------------------------------------------------------------
//  dealloc:
//      Destructor.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void) dealloc
{
    @synchronized( self )
    {
        if( iconRef != NULL )
        {
            (OSErr) ReleaseIconRef( iconRef );
            iconRef = NULL;
        }
        [cachedImage release];
        cachedImage = nil;
        
        #if IKICON_DEBUG
        if( !dummy )
            NSLog(@"*** IKIcon %@ released", UKDebugNameFor(self));
        #endif
    }
    [super dealloc];
}


// -----------------------------------------------------------------------------
//  size:
//      Return the size of this icon. Defaults to 128x128 for icons whose
//      cached NSImage hasn't been loaded yet.
// -----------------------------------------------------------------------------

-(NSSize)   size
{
    if( cachedImage )
        return [cachedImage size];
    else
        return NSMakeSize(128,128);     // FIX ME! Should query the IconRef for available sizes!
}


// -----------------------------------------------------------------------------
//  image:
//      Returns an NSImage of our icon ref. This NSImage contains an
//      IKIconRefImageRep, which calls drawRect: on this icon ref to take care
//      of nicely scaling the icon as needed.
//
//      We don't retain this NSImage because it would cause a circle where the
//      ImageRep retains us, and we retain it. And anyway, an iconRef image rep
//      is lightweight.
//
//  REVISIONS:
//      2005-02-11  UK  Changed to use IKIconRefImageRep instead of manually
//                      assembling an NSImage with several reps.
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(NSImage*) image
{
    NSImage*    img = nil;
    
    @synchronized( self )
    {
        NSRect      iconRect = { { 0,0 }, { 128,128 } };
        img = [[NSImage alloc] initWithSize: iconRect.size];
        NSImageRep* iconRep = [[[IKIconRefImageRep alloc] initWithIcon: self] autorelease];
        [img addRepresentation: iconRep];
        
        #if IKICON_DEBUG
        NSLog(@"IconImageRep %@ points to icon %@", UKDebugNameFor(iconRep), UKDebugNameFor(self));
        #endif
        
        [img autorelease];
        
        /* if( !cachedImage )
        {
            NSRect              iconRect = { { 0,0 }, { 128,128 } };
            NSBitmapImageRep*   miniRep = nil;
            NSBitmapImageRep*   smallRep = nil;
            NSBitmapImageRep*   mediumRep = nil;
            
            cachedImage = [[NSImage alloc] initWithSize: iconRect.size];
            [cachedImage lockFocus];
                [[NSGraphicsContext currentContext] setImageInterpolation: NSImageInterpolationHigh];
                [[NSColor clearColor] set];
                iconRect.size = NSMakeSize(32,32);
                NSRectFill(iconRect);
                [self drawInRect: iconRect];
                miniRep = [[[NSBitmapImageRep alloc] initWithFocusedViewRect: iconRect] autorelease];
                iconRect.size = NSMakeSize(64,64);
                NSRectFill(iconRect);
                [self drawInRect: iconRect];
                mediumRep = [[[NSBitmapImageRep alloc] initWithFocusedViewRect: iconRect] autorelease];
                iconRect.size = NSMakeSize(128,128);
                NSRectFill(iconRect);
                [self drawInRect: iconRect];
            [cachedImage unlockFocus];
            
            [cachedImage setPrefersColorMatch: NO];
            
            [cachedImage addRepresentation: miniRep];
            [cachedImage addRepresentation: smallRep];
            [cachedImage addRepresentation: mediumRep];
        }*/
    }
    
    return img;
}

-(NSDictionary*) dictionaryRepresentation
{
    return nil;
}

// -----------------------------------------------------------------------------
//  iconByAddingIcon:toRect:
//      Composite an icon into a particular rect onto this one. This will
//      create a new icon with the composition result.
//
//  REVISIONS:
//      2005-02-09  UK  Changed to return new object instead of changing
//                      current one.
// -----------------------------------------------------------------------------

-(IKIcon*)  iconByAddingIcon: (IKIcon*)src toRect: (NSRect)pos
{
    return [self iconByAddingIcon: src toRect: pos operation: NSCompositeSourceOver fraction: 1.0];
}


// -----------------------------------------------------------------------------
//  iconByAddingIcon:toRect:operation:fraction:
//      Composite an icon into a particular rect onto this one. This will
//      create a new icon with the composition result.
//
//  REVISIONS:
//      2005-02-09  UK  Changed to return new object instead of changing
//                      current one.
// -----------------------------------------------------------------------------

-(IKIcon*)  iconByAddingIcon: (IKIcon*)src toRect: (NSRect)pos
                    operation:(NSCompositingOperation)op fraction:(float)delta
{
    IconRef     compositeIconRef = NULL;
    OSErr       err = noErr;
    NSSize      mySize = [self size];
    
    // Icon needs to be scaled/moved before compositing?
    if( pos.origin.x != 0 || pos.origin.y != 0
        || pos.size.width != mySize.width || pos.size.height != mySize.height
        || op != NSCompositeCopy || delta != 1.0 )
    {
        NSImage*    img = [[[NSImage alloc] initWithSize: mySize] autorelease];
        [img lockFocus];
            [self drawInRect: NSMakeRect(0,0,mySize.width,mySize.height)];
            if( op != NSCompositeSourceOver || delta != 1.0 )
            {
                NSRect      srcBox = { {0,0}, {0,0} };
                srcBox.size = [src size];
                [[src image] drawInRect: pos fromRect: srcBox operation: op fraction:delta];
            }
            else
                [src drawInRect: pos];
        [img unlockFocus];
        compositeIconRef = [img iconRefRepresentation];
    }
    else    // Simple 1:1 compositing:
    {
        @synchronized( self )
        {
            err = CompositeIconRef( iconRef, [src iconRef], &compositeIconRef );
        }
    }
    
    if( err == noErr )
        return [[[IKIcon alloc] initWithIconRef: compositeIconRef] autorelease];    // Takes over the IconRef.
    else
        return nil;
}


// -----------------------------------------------------------------------------
//  badgeRectForPosition:
//      Return the rect in which a particular badge should be composited onto
//      this icon.
// -----------------------------------------------------------------------------

-(NSRect)       badgeRectForPosition: (IKBadgePosition)pos
{
    NSRect      box = { { 0,0 }, { 0,0 } };
    NSSize      fullSize = [self size];
    
    // If it's a special semantic position, change that into physical:
    if( (pos & IKBadgePositionFlagSemantic) == IKBadgePositionFlagSemantic )
    {
        switch( (int) pos ) // cast shuts off warning about other enum constants.
        {
            case IKBadgePositionLink:
                pos = IKBadgePositionBottomLeft;
                break;

            case IKBadgePositionScript:
                pos = IKBadgePositionBottomLeft;
                break;

            case IKBadgePositionLocked:
                pos = IKBadgePositionBottomLeft;
                break;

            case IKBadgePositionReadOnly:
            case IKBadgePositionWriteOnly:
            case IKBadgePositionStandardReadOnly:
            case IKBadgePositionStandardWriteOnly:
                pos = IKBadgePositionBottomRight;
                break;

            case IKBadgePositionDocumentSubIcon:
            case IKBadgePositionStandardDocumentSubIcon:    // There is no standard document sub-icon yet.
                pos = IKBadgePositionCenter;
                break;

            case IKBadgePositionPluginSubIcon:
            case IKBadgePositionStandardPluginSubIcon:      // There is no standard plugin sub-icon yet.
                pos = IKBadgePositionRight;
                break;
            
            // On Mac, the symlink and permission icons are full-sized with lots of transparent space:
            case IKBadgePositionStandardLink:
            case IKBadgePositionStandardLocked:
            case IKBadgePositionStandardScript:
                pos = IKBadgePositionNone;    // Composite at full size.
                break;
        }
    }
    
    if( pos == IKBadgePositionNone )  // No positioning, just slap on top of the other.
        return NSMakeRect( 0, 0, fullSize.width, fullSize.height );
    
    // Now, make the icon quarter size and nudge it to the right position:
    box.size.width = truncf(fullSize.width / 2);
    box.size.height = truncf(fullSize.height / 2);
    
    if( (pos & IKBadgePositionFlagTop) == IKBadgePositionFlagTop )      // Move to top?
        box.origin.y += fullSize.height -box.size.height;
    if( (pos & IKBadgePositionFlagRight) == IKBadgePositionFlagRight )  // Move to right?
        box.origin.x += fullSize.width -box.size.width;
    
    if( pos == IKBadgePositionBottom || pos == IKBadgePositionTop || pos == IKBadgePositionCenter )    // Horizontally centered?
        box.origin.x += truncf((fullSize.width -box.size.width) /2);
    if( pos == IKBadgePositionLeft || pos == IKBadgePositionRight || pos == IKBadgePositionCenter )    // Vertically centered?
        box.origin.y += truncf((fullSize.height -box.size.height) /2);
    
    return box;
}

// For theme-switching:
-(void)         update
{
    @synchronized( sIKIconObjectCache ) // We don't access the cache directly, but this makes sure we don't call into Icon Services from several threads at once.
    {
        (OSErr) UpdateIconRef( iconRef );
    }
    @synchronized( self )
    {
        [cachedImage autorelease];
        cachedImage = nil;
    }
    
    [[NSNotificationCenter defaultCenter] postNotificationName: IKIconChangedNotification object: self];
}

@end


// -----------------------------------------------------------------------------
//  NSStringFromIconIdentifier:
//      Return an NSString for saving to disk that corresponds to the specified
//      icon identifier. Counterpart to IKIconIdentifierFromString.
//
//  REVISIONS:
//      2005-02-15  UK  Expanded documentation.
// -----------------------------------------------------------------------------

NSString*   NSStringFromIconIdentifier( IKIconIdentifier ident )
{
    NSDictionary*       iconIdentifiers = [NSDictionary dictionaryWithObjectsAndKeys:
                                                @"GenericFolder", [NSNumber numberWithUnsignedLong: kGenericFolderIcon],
                                                @"PrivateFolder", [NSNumber numberWithUnsignedLong: kPrivateFolderIcon],
                                                @"WriteOnlyFolder", [NSNumber numberWithUnsignedLong: kDropFolderIcon],
                                                @"RecyclerFolder", [NSNumber numberWithUnsignedLong: kTrashIcon],
                                                @"TrashFolderFull", [NSNumber numberWithUnsignedLong: kFullTrashIcon],
                                                @"GenericDocument", [NSNumber numberWithUnsignedLong: kGenericDocumentIcon],
                                                @"GenericApplication", [NSNumber numberWithUnsignedLong: kGenericApplicationIcon],
                                                @"GenericPlugIn", [NSNumber numberWithUnsignedLong: kGenericExtensionIcon],
                                                @"LinkBadge", [NSNumber numberWithUnsignedLong: kAliasBadgeIcon],
                                                @"ReadOnlyBadge", [NSNumber numberWithUnsignedLong: kSharingPrivsReadOnlyIcon],
                                                @"ReadWriteBadge", [NSNumber numberWithUnsignedLong: kSharingPrivsReadWriteIcon],
                                                @"WriteOnlyBadge", [NSNumber numberWithUnsignedLong: kSharingPrivsWritableIcon],
                                            nil];
    NSNumber*           key = [NSNumber numberWithUnsignedLong: ident.type];
    NSString*           type = [iconIdentifiers objectForKey: key];
    
    if( type != nil )
        return type;    // Return the cross-platform name for the icon that we found.
    else
        return [NSString stringWithFormat: @"IconServices<%lx:%lx>", ident.creator, ident.type];  // As a last resort, use some platform-specific string.
}


// -----------------------------------------------------------------------------
//  IKIconIdentifierFromString:
//      Return an IKIconIdentifier that corresponds to the specified string.
//      Counterpart to NSStringFromIconIdentifier.
//
//  REVISIONS:
//      2005-02-15  UK  Created.
// -----------------------------------------------------------------------------

IKIconIdentifier    IKIconIdentifierFromString( NSString* str )
{
    IKIconIdentifier    ident = { kSystemIconsCreator, 0 };
    NSDictionary*       iconIdentifiers = [NSDictionary dictionaryWithObjectsAndKeys:
                                                [NSNumber numberWithUnsignedLong: kGenericFolderIcon], @"GenericFolder",
                                                [NSNumber numberWithUnsignedLong: kPrivateFolderIcon], @"PrivateFolder",
                                                [NSNumber numberWithUnsignedLong: kDropFolderIcon], @"WriteOnlyFolder",
                                                [NSNumber numberWithUnsignedLong: kTrashIcon], @"RecyclerFolder",
                                                [NSNumber numberWithUnsignedLong: kFullTrashIcon], @"TrashFolderFull",
                                                [NSNumber numberWithUnsignedLong: kGenericDocumentIcon], @"GenericDocument",
                                                [NSNumber numberWithUnsignedLong: kGenericApplicationIcon], @"GenericApplication",
                                                [NSNumber numberWithUnsignedLong: kGenericExtensionIcon], @"GenericPlugIn",
                                                [NSNumber numberWithUnsignedLong: kAliasBadgeIcon], @"LinkBadge",
                                                [NSNumber numberWithUnsignedLong: kSharingPrivsReadOnlyIcon], @"ReadOnlyBadge",
                                                [NSNumber numberWithUnsignedLong: kSharingPrivsReadWriteIcon], @"ReadWriteBadge",
                                                [NSNumber numberWithUnsignedLong: kSharingPrivsWritableIcon], @"WriteOnlyBadge",
                                            nil];
    NSNumber*           type = [iconIdentifiers objectForKey: str];
    
    if( type != nil )
        ident.type = [type unsignedLongValue];
    else
        ident.creator = 0;
    
    return ident;
}

