/* =============================================================================
    PROJECT:    Filie
    FILE:       UKFSItemController.m
    
    COPYRIGHT:  (c) 2004 by M. Uli Kusterer, all rights reserved.
    
    AUTHORS:    M. Uli Kusterer - UK
    
    LICENSES:   GNU GPL
    
    REVISIONS:
        2004-04-15  UK  Created.
   ========================================================================== */

// -----------------------------------------------------------------------------
//  Headers:
// -----------------------------------------------------------------------------

#import "UKFSItemController.h"


// -----------------------------------------------------------------------------
//  Globals:
// -----------------------------------------------------------------------------

static UKFSItemControllerRegistry*  gRegistrySingleton = nil;


@implementation UKFSItemControllerRegistry

// -----------------------------------------------------------------------------
//  sharedRegistry:
//      Return the singleton object that is our Controller registry.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

+(id)   sharedRegistry
{
	if( gRegistrySingleton == nil )
		return [[UKFSItemControllerRegistry alloc] init];
	else
		return gRegistrySingleton;
}


// -----------------------------------------------------------------------------
//  init:
//      Create a new registry object. Since this is a singleton, this may return
//      NIL if there already is an object. Otherwise, this will set itself up
//      as the singleton.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(id)   init
{
	if( gRegistrySingleton != nil )
	{
		[self release];
		return nil;
	}
	
	self = [super init];
	if( self )
	{
		controllers = [[NSMutableDictionary alloc] init];
		controllerInstances = [[NSMutableArray alloc] init];
		gRegistrySingleton = self;
	}
	
	return self;
}


// -----------------------------------------------------------------------------
//  dealloc:
//      Destructor. Also sets the gRegistrySingleton global to NIL.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void) dealloc
{
    gRegistrySingleton = nil;

	[controllerInstances release];
	[controllers release];
	
	[super dealloc];
}


// -----------------------------------------------------------------------------
//  registerController:forType:
//      Add a new controller to our registry, for viewing the specified type.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void) registerController: (Class)vcls forType: (NSString*)type
{
	[controllers setObject: vcls forKey: type];
}


// -----------------------------------------------------------------------------
//  controllerClassForType:
//      Return the class of object to create for viewing an object of the
//      specified type.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(Class)	controllerClassForType: (NSString*)type
{
	return [controllers objectForKey: type];
}


// -----------------------------------------------------------------------------
//  controllerForItemAtURL:
//      Create a controller object (or return an existing one) for the specified
//      item (URL).
//
//      FIX ME! This assumes a file: URL right now. We need to change the
//      registry to work with URLs!
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(id<UKFSItemController>)		controllerForItemAtURL: (NSURL*)url
{
    return [self controllerForItemAtPath: [url path]];
}


// -----------------------------------------------------------------------------
//  controllerIndexForPath:
//      Return the index in our array of controllers for the controller for the
//      specified path. If there is none, returns NSNotFound.
//
//      This is a workaround for indexOfObject:, which somehow sometimes returns
//      NSNotFound even though there *is* a controller in the list.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(int)  controllerIndexForPath: (NSString*)path
{
    int                 idx = 0;
    NSEnumerator*       enny = [controllerInstances objectEnumerator];
    id<UKFSItemController>  vw;
    
    while( (vw = [enny nextObject]) )
    {
        if( [vw isEqual: path] )
            return idx;
        
        idx++;
    }
    
    return NSNotFound;
}


// -----------------------------------------------------------------------------
//  controllerForItemAtPath:
//      Return the controller for the item at the specified path. Creates a new
//      controller if there's none yet/anymore.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(id<UKFSItemController>)		controllerForItemAtPath: (NSString*)fpath
{
	id<UKFSItemController>  fsiv = nil;
	int					objIndex;
	
	objIndex = [self controllerIndexForPath: fpath];    // controllerInstances indexOfObject: fails to work occasionally, so we do it manually.
	if( objIndex != NSNotFound )
		fsiv = [controllerInstances objectAtIndex: objIndex];
	else
	{
		NSString*		fileType = nil;
		
		fileType = NSHFSTypeOfFile(fpath);
		if( [@"''" isEqualToString: fileType] ) // No OSType?
			fileType = [fpath pathExtension];
		
		if( fileType == nil )
			fileType = @"";
		
		fsiv = [[[UKFSItemControllerRegistry sharedRegistry] controllerClassForType: fileType] controllerForItemAtPath: fpath];
		if( fsiv )
			[controllerInstances addObject: fsiv];
		else
			[[NSWorkspace sharedWorkspace] openFile: fpath];
	}
	
	return fsiv;
}


// -----------------------------------------------------------------------------
//  controllerClosed:
//      Called by controllers when they close so we can remove them from our list,
//      which also releases them.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void)		controllerClosed: (id<UKFSItemController>)fsiv
{
	[controllerInstances removeObject: fsiv];
}


// -----------------------------------------------------------------------------
//  closeAllControllers:
//      Close all controllers that are currently open, e.g. when we quit.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void)		closeAllControllers
{
	NSEnumerator*		enny = [controllerInstances objectEnumerator];
	id<UKFSItemController>  fsiv = nil;
	
	while( (fsiv = [enny nextObject]) )
	{
		[fsiv closeController];
	}
}


// -----------------------------------------------------------------------------
//  memorizeOpenControllers:
//      Save a list of the paths for which viuewers are currently open to the
//      user defaults so they can later be reopened using
//      reopenMemorizedControllers.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void)		memorizeOpenControllers
{
	NSEnumerator*		enny = [controllerInstances objectEnumerator];
	id<UKFSItemController>  fsiv = nil;
	NSMutableArray*		openVs = [NSMutableArray array];
	
	while( (fsiv = [enny nextObject]) )
		[openVs addObject:[fsiv path]];

	[[NSUserDefaults standardUserDefaults] setObject: openVs forKey: @"UKFilieMemorizedControllers"];
}


// -----------------------------------------------------------------------------
//  reopenMemorizedControllers:
//      Reopen the controllers for the paths that have been saved to the user
//      defaults using memorizeOpenControllers. This is useful for reopening the
//      windows the user had open when they quit Filie.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(void)		reopenMemorizedControllers
{
	NSArray*			openVs = [[NSUserDefaults standardUserDefaults] objectForKey: @"UKFilieMemorizedControllers"];
	NSEnumerator*		enny = [openVs objectEnumerator];
	NSString*			vPath = nil;
	
	while( (vPath = [enny nextObject]) )
		[self controllerForItemAtPath: vPath];
}


// -----------------------------------------------------------------------------
//  canCloseAllControllers:
//      Ask all controllers whether they want to close. If a controller needs to be
//      saved before it can be closed, it will return NO and then this will
//      return NO as well. Used when the app quits to give the controllers a chance
//      to save.
//
//  REVISIONS:
//      2004-12-22  UK  Documented.
// -----------------------------------------------------------------------------

-(BOOL)					canCloseAllControllers
{
	NSEnumerator*		enny = [controllerInstances objectEnumerator];
	id<UKFSItemController>  fsiv = nil;
	
	while( (fsiv = [enny nextObject]) )
	{
		if( ![fsiv shouldCloseController] )
			return NO;
	}
	
	return YES;
}


@end
